unit RESPONSE; {Unit of SEASCAPE.PAS}

interface

  uses
 Dialogs, GLOBALS, MATHS, {UTILS,} TIMEFILT, SysUtils;

  procedure MakeSigmaVec (var sigmaVec: seasVector;
                  var s: integer);

  procedure MakeRhoVec (var rhoVec: seasVector;
                  var rho: real;
                  var s: integer);

  procedure QuarterlyQuads (var sigmaVec, rhoVec: seasVector;
                  var rho, zeta: real;
                  var quadNo: integer);

  procedure CompoundQuads (var sigmaVec, rhoVec: seasVector;
                  var rho, zeta: real;
                  var quadNo: integer);

  procedure NewQuarterlyQuads (var sigmaVec, rhoVec: seasVector;
                  var zeta: vector;
                  var rho: real;
                  var quadNo: integer);

  procedure NewCompoundQuads (var sigmaVec, rhoVec: seasVector;
                  var zeta: vector;
                  var rho: real;
                  var quadNo: integer);

  procedure WKCoefficients (var phi, theta: seasVector;
                  q: integer;
                  var lambda: real);

  function BinomialGain (var filterSpec: filterRecord;
                 omega: real): real;

  function TrendCycleResponse (var filterSpec: filterRecord;
                  var phi, theta: seasVector;
                  var s: integer;
                  var omega: real): real;

  function SeasResponse (var phi, theta: seasVector;
                  var s: integer;
                  var omega: real): real;

  procedure GetBinomialWeights (var FilterSpec: FilterRecord;
                  var fIRVec: seasVector;
                  var q: integer);

  function FIRFilterGain (var filterSpec: FilterRecord;
                  var firVec: seasVector;
                  omega: real): real;

  function DualSeaResponse (var phi1, phi2, theta1, theta2: seasVector;
                  var s: integer;
                  var omega: real): real;

  function TripleSeaResponse (var phi0, phi1, phi2, theta0, theta1, theta2: seasVector;
                  var s: integer;
                  var omega: real): real;

  procedure GetFilterGainVector (var filterSpec: filterRecord;
                  var SCREENBOX: IBOX;
                  var spectraVec: longvector;
                  var spectraBOX: rBOX;
                  var phi, theta, phi1, theta1, phi2, theta2: seasVector);

{++++++++++++++++++++++++++++++++++++++++}
  procedure CentralFilterVec (var filterSpec: FilterRecord;
                  var rhoVec, sigmaVec: seasVector;
                   var phi, theta: seasVector);

  procedure SmoothingFilterSpec (var filterSpec: FilterRecord;
                  var rhoVec, sigmaVec, phi, theta, theta2, firVec: seasVector;
                  var spectraVec: longvector;
                  var spectraBOX: rBOX;
                  var SCREENBOX: IBOX);

  procedure TwoPassFilterSpec (var filterSpec: FilterRecord;
                  var rhoVec, sigmaVec1, sigmaVec2: seasVector;
                  var phi1, theta1, phi2, theta2: seasVector);

  procedure OffsetFilterVecs (var filterSpec: FilterRecord;
                  var rhoVec, sigmaVec1, sigmaVec2: seasVector;
                  var phi1, theta1, phi2, theta2: seasVector);

  procedure TripleFilterSpec (var filterSpec: FilterRecord;
                  var rhoVec, sigmaVec, sigmaVec1, sigmaVec2: seasVector;
                  var phi, theta, phi1, theta1, phi2, theta2: seasVector);

implementation

 {_________________________________________________}

  procedure MakeSigmaVec (var sigmaVec: seasVector;
                  var s: integer);

{Ths produces the coefficients of the unit root polynonial Sigma(z) }

    var
      i: integer;
  begin
    for i := 0 to s - 1 do
      sigmaVec[i] := 1.0;
  end; {MakeSigmaVec}

{_________________________________________________}

  procedure MakeRhoVec (var rhoVec: seasVector;
                  var rho: real;

{Ths produces the coefficients of the denominator  polynonial Rho(z)}
{which has to be compouned to produce  teh W-K denominator  }
{phi ( z ) = Sigma(z)Sigma(z^[-1]) + lambda*Rho(z)Rho(z^[-1])  }
                  var s: integer);
    var
      i: integer;
      rhoPower: real;

  begin

    rhoPower := 1.0;
    for i := 0 to s - 1 do
      begin
        rhoVec[i] := rhoPower;
        rhoPower := rhoPower * rho;
      end;
  end; {MakeRhoVec}

{_________________________________________________}

  procedure QuarterlyQuads (var sigmaVec, rhoVec: seasVector;
                  var rho, zeta: real;
                  var quadNo: integer);

{This procedure is for generating the coefficents of polynomials Sigma(z)}
{and Rho(z) for a QUARTERLY filter with the same unique offset from each}
{of the seasonal frequencies. It is used for the Dual Filter}

    var
      quadZeros, quadPoles: seasMatrix;
      omega: seasVector;
      degree: extended;
      alpha: seasVector;
      j, p, k: integer;
      zetaR: real;

  begin

    degree := pi / 180;
    omega[1] := pi / 2.0;
    omega[2] := pi;

    zetaR := zeta * degree;

    QuadZeros[1, 0] := 1.0;
    QuadZeros[1, 1] := -2.0 * Cos(omega[1] + zetaR);
    QuadZeros[1, 2] := 1.0;

    QuadZeros[2, 0] := 1.0;
    if filterSpec.filterType = doubleFilter then
      begin
        QuadZeros[2, 1] := -2.0 * Cos(pi + zetaR);
        QuadZeros[2, 2] := 1.0;
      end;
    if filterSpec.filterType = singleFilter then
      begin
        QuadZeros[1, 1] := 1.0;
        QuadZeros[1, 2] := 0.0;
      end;

    sigmaVec[0] := QuadZeros[1, 0];
    sigmaVec[1] := QuadZeros[1, 1];
    sigmaVec[2] := QuadZeros[1, 2];

    p := 2;
    k := 2; {or is it unity?}
    for j := 2 to quadNo do
      begin
        alpha[0] := quadZeros[j, 0];
        alpha[1] := quadZeros[j, 1];
        alpha[2] := quadZeros[j, 2];
        Convolution(alpha, sigmaVec, p, k);
        k := k + 2
      end;

{++++++++++++++++++++++++++++++++++++++}
    QuadPoles[1, 0] := 1.0;
    QuadPoles[1, 1] := -2.0 * rho * Cos(omega[1] + zetaR);
    QuadPoles[1, 2] := rho * rho;

    QuadPoles[2, 0] := 1.0;
    if filterSpec.filterType = DoubleFilter then
      begin
        QuadPoles[2, 1] := -2.0 * rho * Cos(pi + zetaR);
        QuadPoles[2, 2] := rho * rho;
      end;
    if filterSpec.filterType = singleFilter then
      begin
        QuadPoles[2, 1] := rho;
        QuadPoles[2, 2] := 0.0;
      end;

    rhoVec[0] := quadPoles[1, 0];
    rhoVec[1] := quadPoles[1, 1];
    rhoVec[2] := quadPoles[1, 2];

    p := 2;
    k := 2; {or is it unity?}
    for j := 2 to quadNo do
      begin
        alpha[0] := QuadPoles[j, 0];
        alpha[1] := QuadPoles[j, 1];
        alpha[2] := QuadPoles[j, 2];
        Convolution(alpha, rhoVec, p, k);
        k := k + 2
      end;

  end; {QuarterlyQuads}

{_________________________________________________}

  procedure CompoundQuads (var sigmaVec, rhoVec: seasVector;
                  var rho, zeta: real;
                  var quadNo: integer);

{This procedure is for generating the coefficents of polynomials Sigma(z)}
{and Rho(z) for a MONTHLY filter with the same unique offset from each}
{of the seasonal frequencies. It is used for the Dual Filter}

    var
      quadZeros, quadPoles: seasMatrix;
      omega: seasVector;
      degree: extended;
      alpha: seasVector;
      j, p, k: integer;
      zetaR: real;

  begin

    degree := pi / 180;
    omega[1] := pi / 6.0;
    omega[2] := pi / 3.0;
    omega[3] := pi / 2.0;
    omega[4] := 2.0 * pi / 3.0;
    omega[5] := 5.0 * pi / 6.0;

    zetaR := zeta * degree;

{These relate to the zero frequency and they are not active}
    QuadZeros[0, 0] := 1.0;
    QuadZeros[0, 1] := -1.0;
    QuadZeros[0, 2] := 0.0;

    for j := 1 to 5 do
      begin
        QuadZeros[j, 0] := 1.0;
        QuadZeros[j, 1] := -2.0 * Cos(omega[j] + zetaR);
        QuadZeros[j, 2] := 1.0;
      end;

    QuadZeros[6, 0] := 1.0;
    if filterSpec.filterType = DoubleFilter then
      begin
        QuadZeros[6, 1] := -2.0 * Cos(pi + zetaR);
        QuadZeros[6, 2] := 1.0;
      end;
    if filterSpec.filterType = singleFilter then
      begin
        QuadZeros[6, 1] := 1.0;
        QuadZeros[6, 2] := 0.0;
      end;

    sigmaVec[0] := QuadZeros[1, 0];
    sigmaVec[1] := QuadZeros[1, 1];
    sigmaVec[2] := QuadZeros[1, 2];

    p := 2;
    k := 2; {or is it unity?}
    for j := 2 to quadNo do
      begin
        alpha[0] := quadZeros[j, 0];
        alpha[1] := quadZeros[j, 1];
        alpha[2] := quadZeros[j, 2];
        Convolution(alpha, sigmaVec, p, k);
        k := k + 2
      end;

{++++++++++++++++++++++++++++++++++++++}

   {These are not active}
    quadPoles[0, 0] := 1.0;
    quadPoles[0, 1] := -rho;
    quadPoles[0, 2] := 0.0;

    for j := 1 to 5 do
      begin
        quadPoles[j, 0] := 1.0;
        quadPoles[j, 1] := -2.0 * rho * Cos(omega[j] + zetaR);
        quadPoles[j, 2] := rho * rho;
      end;

    quadPoles[6, 0] := 1.0;
    if filterSpec.filterType = DoubleFilter then
      begin
        quadPoles[6, 1] := -2.0 * rho * Cos(pi + zetaR);
        quadPoles[6, 2] := rho * rho;
      end;
    if filterSpec.filterType = singleFilter then
      begin
        quadPoles[6, 1] := rho;
        quadPoles[6, 2] := 0.0;
      end;

    rhoVec[0] := quadPoles[1, 0];
    rhoVec[1] := quadPoles[1, 1];
    rhoVec[2] := quadPoles[1, 2];

    p := 2;
    k := 2;
    for j := 2 to quadNo do
      begin
        alpha[0] := quadPoles[j, 0];
        alpha[1] := quadPoles[j, 1];
        alpha[2] := quadPoles[j, 2];
        Convolution(alpha, rhoVec, p, k);
        k := k + 2
      end;

    noModel := false;

  end; {CompoundQuads}
{_________________________________________________}

  procedure NewQuarterlyQuads (var sigmaVec, rhoVec: seasVector;
                  var zeta: vector;  {zetaSup or zetaSub}
                  var rho: real;
                  var quadNo: integer);

{This procedure is for generating the coefficents of polynomials Sigma(z)}
{and Rho(z) for a QUARTERLY filter with different offsets from each}
{of the seasonal frequencies. It is used for the Triple Filter}

    var
      quadZeros, quadPoles: seasMatrix;
      omega: seasVector;
      degree: extended;
      zetaR: real;
      alpha: seasVector;
      j, p, k: integer;

  begin
      {SHOWMESSAGE(' In NewQuarterlyQuads zetaSub[1] = ' + FloatToStr(zeta[1]));}

    degree := pi / 180;
    omega[1] := pi / 2.0;
    omega[2] := pi;

    zetaR := zeta[1] * degree;
    QuadZeros[1, 0] := 1.0;
    QuadZeros[1, 1] := -2.0 * Cos(omega[1] + zetaR);
    QuadZeros[1, 2] := 1.0;

    QuadZeros[2, 0] := 1.0;
    if offset = true then
      begin
        zetaR := zeta[2] * degree;
        QuadZeros[2, 1] := -2.0 * Cos(pi + zetaR);
        QuadZeros[2, 2] := 1.0;
      end
    else
      begin
        QuadZeros[2, 1] := 1.0;
        QuadZeros[2, 2] := 0.0;
      end;

    sigmaVec[0] := QuadZeros[1, 0];
    sigmaVec[1] := QuadZeros[1, 1];
    sigmaVec[2] := QuadZeros[1, 2];

    p := 2;
    k := 2;
    for j := 2 to quadNo do
      begin
        alpha[0] := quadZeros[j, 0];
        alpha[1] := quadZeros[j, 1];
        alpha[2] := quadZeros[j, 2];
        Convolution(alpha, sigmaVec, p, k);
        k := k + 2
      end;

{++++++++++++++++++++++++++++++++++++++}

    zetaR := zeta[1] * degree;
    quadPoles[1, 0] := 1.0;
    quadPoles[1, 1] := -2.0 * rho * Cos(omega[1] + zetaR);
    quadPoles[1, 2] := rho * rho;


    quadPoles[2, 0] := 1.0;
    if offset then
      begin
        zetaR := zeta[2] * degree;
        quadPoles[2, 1] := -2.0 * rho * Cos(omega[2] + zetaR);
        quadPoles[2, 2] := rho * rho;
      end
    else
      begin
        quadPoles[2, 1] := rho;
        quadPoles[2, 2] := 0.0;
      end;

    rhoVec[0] := quadPoles[1, 0];
    rhoVec[1] := quadPoles[1, 1];
    rhoVec[2] := quadPoles[1, 2];

    p := 2;
    k := 2;
    for j := 2 to quadNo do
      begin
        alpha[0] := quadPoles[j, 0];
        alpha[1] := quadPoles[j, 1];
        alpha[2] := quadPoles[j, 2];
        Convolution(alpha, rhoVec, p, k);
        k := k + 2
      end;

    noModel := false;


  end; {NewQuarterlyQuads}

{_________________________________________________}

  procedure NewCompoundQuads (var sigmaVec, rhoVec: seasVector;
                  var zeta: vector;
                  var rho: real;
                  var quadNo: integer);

{This procedure is for generating the coefficents of polynomials Sigma(z)}
{and Rho(z) for a MONTHLY filter with different offsets from each}
{of the seasonal frequencies. It is used for the Triple Filter}

    var
      quadZeros, quadPoles: SeasMatrix;
      omega: seasVector;
      degree: extended;
      zetaR: reaL;
      alpha: seasVector;
      j, p, k: integer;

  begin

    degree := pi / 180;
    omega[1] := pi / 6.0;
    omega[2] := pi / 3.0;
    omega[3] := pi / 2.0;
    omega[4] := 2.0 * pi / 3.0;
    omega[5] := 5.0 * pi / 6.0;

    QuadZeros[0, 0] := 1.0;
    QuadZeros[0, 1] := -1.0;
    QuadZeros[0, 2] := 0.0;

    for j := 1 to 5 do
      begin
        zetaR := zeta[j] * degree;
        QuadZeros[j, 0] := 1.0;
        QuadZeros[j, 1] := -2.0 * Cos(omega[j] + zetaR);
        QuadZeros[j, 2] := 1.0;
      end;

    QuadZeros[6, 0] := 1.0;
    if offset = true then
      begin
        zetaR := zeta[6] * degree;
        QuadZeros[6, 1] := -2.0 * Cos(pi + zetaR);
        QuadZeros[6, 2] := 1.0;
      end
    else
      begin
        QuadZeros[6, 1] := 1.0;
        QuadZeros[6, 2] := 0.0;
      end;

    sigmaVec[0] := QuadZeros[1, 0];
    sigmaVec[1] := QuadZeros[1, 1];
    sigmaVec[2] := QuadZeros[1, 2];

    p := 2;
    k := 2;
    for j := 2 to quadNo do
      begin
        alpha[0] := quadZeros[j, 0];
        alpha[1] := quadZeros[j, 1];
        alpha[2] := quadZeros[j, 2];
        Convolution(alpha, sigmaVec, p, k);
        k := k + 2
      end;

{++++++++++++++++++++++++++++++++++++++}

    quadPoles[0, 0] := 1.0;
    quadPoles[0, 1] := -rho;
    quadPoles[0, 2] := 0.0;

    for j := 1 to 5 do
      begin
        zetaR := zeta[j] * degree;
        quadPoles[j, 0] := 1.0;
        quadPoles[j, 1] := -2.0 * rho * Cos(omega[j] + zetaR);
        quadPoles[j, 2] := rho * rho;
      end;

    quadPoles[6, 0] := 1.0;
    if offset then
      begin
        zetaR := zeta[6] * degree;
        quadPoles[6, 1] := -2.0 * rho * Cos(pi + zetaR);
        quadPoles[6, 2] := rho * rho;
      end
    else
      begin
        quadPoles[6, 1] := rho;
        quadPoles[6, 2] := 0.0;
      end;

    rhoVec[0] := quadPoles[1, 0];
    rhoVec[1] := quadPoles[1, 1];
    rhoVec[2] := quadPoles[1, 2];

    p := 2;
    k := 2;
    for j := 2 to quadNo do
      begin
        alpha[0] := quadPoles[j, 0];
        alpha[1] := quadPoles[j, 1];
        alpha[2] := quadPoles[j, 2];
        Convolution(alpha, rhoVec, p, k);
        k := k + 2
      end;

    noModel := false;

  end; {NewCompoundQuads}

{_________________________________________________}

  procedure WKCoefficients (var phi, theta: seasVector;
                  q: integer;
                  var lambda: real);

{This procedure forms the Wiener-Kolmogorov coefficients from the parameters}
{phi and theta, and it normalises the gain of the W-K filter at zero frequency.}
{The procedure is used in <CentralFilterVec>, <TwoPassFilterSpec> and <OffsetFilterVecs>.}
{phi decomes the denominator of the W-K filter}

    var
      j: integer;
      thetaSum, phiSum, factor: extended;

  begin{Make lambda* R(z)R(1/z) + S(z)S(1/z)}

    for j := 0 to q do
      phi[j] := lambda * phi[j] + theta[j];

    thetaSum := theta[0];
    phiSum := phi[0];
    for j := 1 to q do
      begin
        thetaSum := thetaSum + 2 * theta[j];
        phiSum := phiSum + 2 * phi[j];
      end;
    factor := thetaSum / phiSum;

    for j := 0 to q do
      phi[j] := phi[j] * factor;

  end; {WKCoefficients}

{_________________________________________________}

  function SeasResponse (var phi, theta: seasVector;
                  var s: integer;
                  var omega: real): real;
    var
      numerator, denominator: extended;
      j: integer;

  begin
    numerator := theta[0];
    denominator := phi[0];
    for j := 1 to s - 1 do
      begin
        numerator := numerator + 2 * theta[j] * cos(j * omega);
        denominator := denominator + 2 * phi[j] * cos(j * omega);
      end;
    SeasResponse := numerator / denominator;

  end; {SeasResponse}

  {______________________________________________________________________________}

    procedure GetBinomialWeights (var filterSpec: FilterRecord;
                    var fIRVec: seasVector;
                    var q: integer);

  {This uses a recursive algorithm for generating the binomial ordinates}

      var
        j: integer;

    begin
      q := (filterSpec.FilterSpan -1)div 2;
        fIRVec[0] := Power(0.5, q);
      for j := 1 to q do
        fIRVec[j] := (fIRVec[j - 1] * (q - j + 1)) / j;

    end; {GetBinomialWeights}

{______________________________________________________________________________}

    function FIRFilterGain (var filterSpec: FilterRecord;
                    var firVec: seasVector;
                    omega: real): real;

   {This is genuinely the gain and not the squared gain}
      var
        gain: real;
        j, m: integer;

    begin
      m := (filterSpec.filterSpan - 1) div 2;
      gain := firVec[0];
      for j := 1 to m do
        gain := gain + 2 * firVec[j] * Cos(omega * j);

      FIRFilterGain := gain;
    end; {FIRFilterGain}

{______________________________________________________________________________}

  function BinomialGain (var filterSpec: filterRecord;
                 omega: real): real;
  var
     gain, x, n: real;

 begin
   n := filterSpec.filterOrder;

   x := Cos(omega / 2.0);
   gain := RealPower(x, n);
   BinomialGain := gain;

 end;{BinomialGain}

{______________________________________________________________________________}

function TrendCycleResponse (var filterSpec: filterRecord;
                var phi, theta: seasVector;
                var s: integer;
                var omega: real): real;
    var
      firstResponse, taperResponse: extended;

  begin
    firstResponse := SeasResponse (phi, theta, s, omega);
    taperResponse :=  BinomialGain (filterSpec, omega);
    TrendCycleResponse := firstResponse * taperResponse;

  end; {TrendCycleResponse }

{_________________________________________________}

  function DualSeaResponse (var phi1, phi2, theta1, theta2: seasVector;
                  var s: integer;
                  var omega: real): real;
    var
      numerator, denominator, firstResponse: extended;
      j: integer;

  begin

    firstResponse := 1.0;

    numerator := theta1[0];
    denominator := phi1[0];
    for j := 1 to s - 2 do {10 originally}
      begin
        numerator := numerator + 2 * theta1[j] * cos(j * omega);
        denominator := denominator + 2 * phi1[j] * cos(j * omega);
      end;
    firstResponse := numerator / denominator;

    numerator := theta2[0];
    denominator := phi2[0];
    for j := 1 to s do {12 originally}
      begin
        numerator := numerator + 2 * theta2[j] * cos(j * omega);
        denominator := denominator + 2 * phi2[j] * cos(j * omega);
      end;
    DualSeaResponse := firstResponse * numerator / denominator;

  end; {DualSeaResponse}

{_________________________________________________}

  function TripleSeaResponse (var phi0, phi1, phi2, theta0, theta1, theta2: seasVector;
                  var s: integer;
                  var omega: real): real;

{phi0, theta0 are for the central filter,}
{ phi1, theta1,phi2, theta2. are for the offset filters}

    var
      numerator, denominator, firstResponse: extended;
      j: integer;

  begin

    firstResponse := 1.0;
    numerator := theta1[0];
    denominator := phi1[0];
    for j := 1 to s - 2 do {10 = s - 2 originally}
      begin
        numerator := numerator + 2 * theta1[j] * cos(j * omega);
        denominator := denominator + 2 * phi1[j] * cos(j * omega);
      end;
    firstResponse := numerator / denominator;

    numerator := theta2[0];
    denominator := phi2[0];
    for j := 1 to s do {s= 12 originally}
      begin
        numerator := numerator + 2 * theta2[j] * cos(j * omega);
        denominator := denominator + 2 * phi2[j] * cos(j * omega);
      end;
    firstResponse := firstResponse * numerator / denominator;

    numerator := theta0[0];
    denominator := phi0[0];
    for j := 1 to s - 1 do {11 =s-1 originally}
      begin
        numerator := numerator + 2 * theta0[j] * cos(j * omega);
        denominator := denominator + 2 * phi0[j] * cos(j * omega);
      end;
    TripleSeaResponse := firstResponse * numerator / denominator;


  end; {TripleSeaResponse}

{_________________________________________________}

  procedure GetFilterGainVector (var filterSpec: filterRecord;
                  var SCREENBOX: IBOX;
                  var spectraVec: longvector;
                  var spectraBOX: rBOX;
                  var phi, theta, phi1, theta1, phi2, theta2: seasVector);

    var
      BOXWDT, s, i: integer;
      yMax, yMin, omega, pi, inc: real;

  begin
    s := filterSpec.seasonality;
    pi := 4.0 * Arctan(1.0);
    BOXWDT := SCREENBOX.X2 - SCREENBOX.X1;

    inc := pi / BOXWDT;
    omega := 0.0;

    yMax := 0.0;
    yMin := 0.0;
    for i := 0 to BOXWDT do
      begin {i}
        if filterSpec.filterType = singleFilter then
          spectraVec[i] := SeasResponse(phi, theta, s, omega);
        if filterSpec.filterType = taperFunction then
           spectraVec[i] := BinomialGain (filterSpec, omega);
        if filterSpec.filterType = TaperedFilter then
          spectraVec[i] := TrendCycleResponse(filterSpec, phi, theta, s, omega);
        if filterSpec.filterType = DoubleFilter then
          spectraVec[i] := DualSeaResponse(phi1, phi2, theta1, theta2, s, omega);
        if filterSpec.filterType = tripleFilter then
          spectraVec[i] := TripleSeaResponse(phi, phi1, phi2, theta, theta1, theta2, s, omega);

        yMax := RMax(yMax, spectraVec[i]);
        yMin := RMin(yMin, spectraVec[i]);
        omega := omega + inc;
      end; {i}

    spectraBOX.NOPUNCT := BOXWDT;
    spectraBOX.xMax := pi;
    spectraBOX.xMin := 0.0;
    spectraBOX.yMax := yMax;
    spectraBOX.yMin := yMin;
    spectraBOX.xRB := pi;
    spectraBOX.xLB := 0.0;
    spectraBOX.yUB := yMax;
    spectraBOX.yLB := yMin;

  end; {GetFilterGainVector}

{_________________________________________________}

  procedure CentralFilterVec (var filterSpec: FilterRecord;
                  var rhoVec, sigmaVec: seasVector;
                  var phi, theta: seasVector);

{observe that spectraVec, spectraBOX, and SCREENBOX are absent from the parameter list);}
{This serves both for the ordinary filter and the ThreePass filter}
{if affects the seaonal adjustment of the data}

    var
      s, j: integer;
      rho,lambda: real;

  begin
    s := filterSpec.seasonality;
    rho := filterSpec.rho;
    lambda := filterSpec.lambda;

    MakeSigmaVec(sigmaVec, s);
    MakeRhoVec(rhoVec, rho, s);
    Laurent(sigmaVec, theta, s - 1);
    Laurent(rhoVec, phi, s - 1);
    WKCoefficients(phi, theta, s - 1, lambda);

  end; {CentralFilterVec}

{_________________________________________________}

  procedure SmoothingFilterSpec (var filterSpec: FilterRecord;
                  var rhoVec, sigmaVec, phi, theta, theta2, firVec: seasVector;
                  var spectraVec: longvector;
                  var spectraBOX: rBOX;
                  var SCREENBOX: IBOX);

    var
      muSum, muNo, lambda, rho, inc, pi, omega: real;
      ymax, yMin: real;
      i, j, s, q, sPlus, BOXWDT: integer;
      F: text;
      legend: string;

  begin
    s := filterSpec.seasonality;
    q := (filterSpec.filterSpan -1) div 2;
    sPlus := s;
    lambda := filterSpec.lambda;
    rho := filterSpec.rho;

    GetBinomialWeights (FilterSpec, fIRVec, q);

    pi := 4.0 * Arctan(1.0);
    BOXWDT := SCREENBOX.X2 - SCREENBOX.X1;

    inc := pi / BOXWDT;
    omega := 0.0;

    yMax := 0.0;
    yMin := 0.0;
    for i := 0 to BOXWDT do
      begin {i}
       spectraVec[i] := FIRFilterGain (filterSpec,firVec, omega);
        yMax := RMax(yMax, spectraVec[i]);
        yMin := RMin(yMin, spectraVec[i]);
       omega := omega + inc;
      end; {i}

    spectraBOX.NOPUNCT := BOXWDT;
{spectraBOX.xRange := BOXWDT;}
    spectraBOX.xMax := pi;
    spectraBOX.xMin := 0.0;
    spectraBOX.yMax := yMax;
    spectraBOX.yMin := yMin;
    spectraBOX.xRB := pi;
    spectraBOX.xLB := 0.0;
    spectraBOX.yUB := yMax;
    spectraBOX.yLB := yMin;


  end;{SmoothingFilterSpec}

{_________________________________________________}

  procedure TwoPassFilterSpec (var filterSpec: FilterRecord;
                  var rhoVec, sigmaVec1, sigmaVec2: seasVector;
                  var phi1, theta1, phi2, theta2: seasVector);

{WE COULD Eliminate   spectraVec, spectraBOX, and SCREENBOX from the parameter list);}
{in this routine, the offsets are symmetric relative to the central seasonal frequency}
{we could use <OffsetFilterVecs> vecs here}

    var
      lambda, rho, zeta: real;
      j, s, sPlus, quadNo: integer;

  begin
    filterSpec.FilterType := doubleFilter;
    s := filterSpec.seasonality;
    lambda := filterSpec.lambda;
    rho := filterSpec.rho;
    zeta := filterSpec.zeta;
    offset := true;

{Observe that with the offsets, there is a quadratic Factor at (pi pm offset)}
{which we choose to exclude from the first filter. i.e from phi1, theta1}
{First Offset Vec: phi1, theta1}

    quadNo := (s - 2) div 2; {1 for quarterly, 5 for monthly ?}
    sPlus := 2 * quadNo; {sPlus is the degree of the filter polynomials}
    zeta := -zeta;

    if s = 4 then
      QuarterlyQuads(sigmaVec1, rhoVec, rho, zeta, quadNo)
    else
      CompoundQuads(sigmaVec1, rhoVec, rho, zeta, quadNo);

    Laurent(sigmaVec1, theta1, s - 2);
    Laurent(rhoVec, phi1, s - 2);
    WKCoefficients(phi1, theta1, sPlus, lambda);

    {Second Offset Vec: phi2, theta2}
    zeta := -zeta;
    quadNo := quadNo + 1;
    sPlus := 2 * quadNo; {sPlus is the degree of the filter polynomials}

    if s = 4 then
      QuarterlyQuads(sigmaVec2, rhoVec, rho, zeta, quadNo)
    else
      CompoundQuads(sigmaVec2, rhoVec, rho, zeta, quadNo);

    Laurent(sigmaVec2, theta2, s);
    Laurent(rhoVec, phi2, s);
    WKCoefficients(phi2, theta2, sPlus, lambda);

  end; {TwoPassFilterSpec}

{_________________________________________________}

  procedure OffsetFilterVecs (var filterSpec: FilterRecord;
                  var rhoVec, sigmaVec1, sigmaVec2: seasVector;
                  var phi1, theta1, phi2, theta2: seasVector);

{This routine serves only for the triple filter}
{it takes zetaSub an zetSup as paramneters albeit that they are not in the headline}

{SEADODOS has the folllowing fix in GettheOffsets:}
  {if i =   quadNo then
    zetaSup[i] - zteaSub[i] }

    var
      {zetaSub, zetaSup: vector;}
      s, quadNo, sPlus: integer;
      rho, lambda: real;

  begin
    s := filterSpec.seasonality;
    rho := filterSpec.rho;
    lambda := filterSpec.lambda;


    quadNo := s div 2;
    sPlus := 2 * quadNo;

    quadNo := (s - 2) div 2;
    sPlus := 2 * quadNo; {twice quadNo }


    {SHOWMESSAGE(' In OffSeFilterVecs zetaSub[1] = ' + FloatToStr(zetaSub[1]));}


    if s = 12 then
      NewCompoundQuads(sigmaVec1, rhoVec, zetaSup, rho, quadNo);
    if s = 4 then
      NewQuarterlyQuads(sigmaVec1, rhoVec, zetaSup, rho, quadNo);

    Laurent(sigmaVec1, theta1, s - 2); {Theta1 is 10 of 1}
    Laurent(rhoVec, phi1, s - 2);
    WKCoefficients(phi1, theta1, sPlus, lambda);

    quadNo := s div 2;
    sPlus := 2 * quadNo;

    if s = 12 then
      NewCompoundQuads(sigmaVec2, rhoVec, zetaSub, rho, quadNo);
    if s = 4 then
      NewQuarterlyQuads(sigmaVec2, rhoVec, zetaSub, rho, quadNo);

    Laurent(sigmaVec2, theta2, s);
    Laurent(rhoVec, phi2, s);
    WKCoefficients(phi2, theta2, s, lambda);

  end; {OffsetFilterVecs }

{_________________________________________________}

  procedure TripleFilterSpec (var filterSpec: FilterRecord;
                  var rhoVec, sigmaVec, sigmaVec1, sigmaVec2: seasVector;
                  var phi, theta, phi1, theta1, phi2, theta2: seasVector);
    var
      quadNo, s, sPlus: integer;
      rho, lambda: real;

  begin
    filterSpec.FilterType := tripleFilter;
    s := filterSpec.seasonality;
    rho := filterSpec.rho;
    lambda := filterSpec.lambda;


    quadNo := (s - 2) div 2; {This will serve for central filter}
    sPlus := 2 * quadNo;

    offset := true;

    CentralFilterVec (filterSpec, rhoVec, sigmaVec, phi, theta);  {in RESPONSE}

    {SHOWMESSAGE(' In TripleFilterSpec zetaSub[1] = ' + FloatToStr(zetaSub[1]));}
    {SHOWMESSAGE(' In TripleFilterSpec zetaSup[1] = ' + FloatToStr(zetaSup[1]));}

    offset := true;
    OffsetFilterVecs(filterSpec, rhoVec, sigmaVec1, sigmaVec2, phi1, theta1, phi2, theta2); {in RESPONSE}


  end; {TripleFilterSpec}
{_________________________________________________}

end. {RESPONSE: unit of SEASCAPE.PAS}
