unit WIENER;  {unit of SEASCAPES}

{$mode objfpc}{$H+}

interface

uses
  GLOBALS, MATHS, Classes, SysUtils;

procedure GetFilterGainVector (var filterSpec: FilterRecord;
                var filterBOX: RBox;
                var SCREENBOX: Ibox;
                var dataVector: longVector);  {in FILTERS}

function DigitalButterGain (var n: integer;
                var cutOff, omega: real): real; {in FILTERS}

function HPGain (omega, lambda: real): real; {in FILTERS}

procedure MakeButterTrend (var dataBOX, trendBOX, residueBOX: rBOX;
                 var filterSpec: filterRecord;
                 var dataVector, trendVector, residueVec: longVector);{in WIENER}

procedure ButterworthFilter (n, Tcap: integer;
                 omegaCut: real;
                 var x: longVector;
                 var y: longVector);    {in FILTERS}

procedure MakeHPTrend (var dataBOX, trendBOX, residueBOX: rBOX;
                var filterSpec: FilterRecord;
                var dataVector, trendVector, residueVec: longVector); {in WIENER}

procedure HPFilter (Tcap: integer;
                  lambda: real;
                  var x: longVector;
                  var y: longVector);   {in FILTERS}

procedure Toeplitz (gamma: vector;
                 var y: longVector;
                 n, q: integer);

procedure SumDifference (var theta: vector;
                n: integer;
                sign: integer);

procedure FormGamma (var gamma: vector;
                 mu: vector;
                 r: integer);

procedure FormMvec (var mVec: vector;
                  n: integer);

procedure FormWvec (var wVec: vector;
                n: integer;
                lamOne, lamTwo: real); {in FILTERS}

procedure FormQuinDiag (var wVec: vector;
                lambda: real);  {in FILTERS}

procedure VariQinDiag (n: integer;
                var d, s, q, y: longvector);

procedure QprimeY (var y: longVector;
                Tcap: integer);  {in FILTERS}

procedure FormQy (var y: longVector;
                  Tcap: integer); {in FILTERS}

procedure FormSVec (var sVec: vector;
                  n: integer); {in FILTERS}

procedure GammaY (gamma: vector;
                  var y: longVector;
                  Tcap, n: integer);  {in FILTERS}

procedure FormTrend (var y: longVector;
                  var x: longVector;
                  lamOne: real;
                  Tcap: integer);  {in FILTERS}

implementation

{______________________________________________________________________________}

 procedure GetFilterGainVector(var filterSpec: FilterRecord;
                 var filterBOX: rBox;
                 var SCREENBOX: Ibox;
                 var dataVector: longVector);

{filterBOX and dataVector are deterermined within this procedure}
{filterBOX should become plotBOX and  dataVector should become plotVector}

   var
     BOXWDT, i, n: integer;
     yMax, yMin, omega, pi, lambda, cutOff, inc: real;
{phi, chi <==> numerator denominator are global vectors}

   begin{filterSpec}
       pi := 4.0 * Arctan(1.0);
       cutOff := filterSpec.cutOff * pi/180.0;
       lambda := filterSpec.smoothingParameter;
       n :=  filterSpec.filterOrder;
       BOXWDT := SCREENBOX.X2 - SCREENBOX.X1;

       inc := pi / BOXWDT;
       omega := 0.0;

       yMax := 0.0;
       yMin := 0.0;

       for i := 0 to BOXWDT do
         begin {i}

           if filterSpec.filterFamily = Butterworth then
             dataVector[i] := DigitalButterGain(n, cutOff, omega);

           if filterSpec.filterFamily = HodrickPrescott then
             dataVector[i] := HPGain(omega, lambda);

           yMax := RMax(yMax, dataVector[i]);
           yMin := RMin(yMin, dataVector[i]);
           omega := omega + inc;
         end; {i}

{Should the be  filterBOX?}
   filterBOX.NOPUNCT := BOXWDT;
   filterBOX.xMax := pi;
   filterBOX.xMin := 0.0;
   filterBOX.yMax := yMax;
   filterBOX.yMin := yMin; {do not set this to zero; Eg the Henderson gain}
   filterBOX.xRB := pi;
   filterBOX.xLB := 0.0;
   filterBOX.yUB := yMax;
   filterBOX.yLB := yMin;

 end; {GetFilterGainVector}

{______________________________________________________________________________}

 function DigitalButterGain (var n: integer;
                 var cutOff, omega: real): real;

{Gain of SquareGain?}
   var
     x, gain: real;
    twoN: integer;

 begin
   x := Tan(omega / 2) * CoTan(CutOff / 2);
   twoN := 2 * n;
   gain := 1 / (1 + Power(x, twoN));
   DigitalButterGain := gain;

 end; {DigitalButterGain}

{______________________________________________________________________________}

function HPGain (omega, lambda: real): real;

  var
    x, gain: real;

begin
  x := 2 * Sin(omega / 2);
  gain := 1 / (1 + lambda * Power(x, 4));
  if (filterSpec.filterType = highpass) then
    gain := 1 - gain;
  HPGain := gain;

end; {HPGain}

{______________________________________________________________________________}

  procedure MakeButterTrend (var dataBOX, trendBOX, residueBOX: rBOX;
                  var filterSpec: FilterRecord;
                  var dataVector, trendVector, residueVec: longVector);

    var
      minResidual, maxResidual, trendMean, residueMean, pi, omegaCut: real;
      i: smallInt;
      x, y: longVector;

  begin
    pi := 4.0 *Arctan(1.0);
    omegaCut  := filterSpec.cutoff * pi/180.0;;
    for i := 0 to dataBOX.NOPUNCT - 1 do
      begin
        x[i] := dataVector[i];
        y[i] := dataVector[i];
      end;

    ButterworthFilter(filterSpec.filterOrder, dataBOX.NOPUNCT, omegaCut, x, y);   {in FILTERS}

   {MakeDataBox(residueBOX, residueVec); MakeDataBox(trendBOX, trendVec);}
    minResidual := y[0];
    maxResidual := y[0];
    trendMean := 0.0;
    residueMean := 0.0;
    for i := 0 to dataBOX.NOPUNCT - 1 do
      begin
        trendVector[i] := x[i];
        residueVec[i] := y[i];
        trendMean := trendMean + trendVector[i];
        residueMean := residueMean + residueVec[i];
        maxResidual := Rmax(maxResidual, y[i]);
        minResidual := Rmin(minResidual, y[i]);
      end;
    residueMean := residueMean / dataBOX.NOPUNCT;
    trendMean := trendMean / dataBOX.NOPUNCT;

    residueBOX := dataBOX;
    trendBOX := dataBOX;
    trendBOX.yMean := trendMean;
    residueBOX.yMean := residueMean;
    residueBOX.yMin := minResidual;
    residueBOX.yMax := maxResidual;

    weHaveATrendCycle := true;
    weHavePolyResiduals := false;
    weHaveTrendCycleResiduals := true;

  end; {MakeButterTrend}

{______________________________________________________________________________}

  procedure ButterworthFilter (n, Tcap: integer;
                  omegaCut: real;
                  var x: longVector;
                  var y: longVector);

    var
      lamOne, lamTwo: real;
      sVec, wVec: vector;

  begin
    lamOne := 1 / Tan(omegaCut / 2);
    lamOne := Power(lamOne, 2 * n);
    lamTwo := 1.0;
    if lamOne > 1.0 then
      begin
        lamTwo := 1 / lamOne;
        LamOne := 1.0;
      end;

    x := y;
    FormWvec(wVec, n, lamOne, lamTwo); { W = M + lambda*Q'SQ }
    QprimeY(x, Tcap); {d = Q'y: we are loosing two points in taking 2nd differences}
    Toeplitz(wVec, x, Tcap - 3, n); {solve (M + lambda*Q'SQ)g = d for g}
    FormQy(x, Tcap); {Form Qg}
    FormSvec(sVec, n - 2);
    GammaY(sVec, x, Tcap, n - 2); {Form SQg}
    FormTrend(y, x, lamOne, Tcap); {x is trend, y is residual}

  end; {ButterworthFilter}

{______________________________________________________________________________}

  procedure MakeHPTrend (var dataBOX, trendBOX, residueBOX: rBOX;
                  var filterSpec: FilterRecord;
                  var dataVector, trendVector, residueVec: longVector);

    var
      minResidual, maxResidual, trendMean, residueMean, lambda: real;
      i: integer;
      x, y: longVector;

  begin
    lambda := filterSpec.smoothingParameter;
    for i := 0 to dataBOX.NOPUNCT - 1 do
      begin
        x[i] := dataVector[i];
        y[i] := dataVector[i];
      end;
    HPFilter(dataBOX.NOPUNCT, lambda, x, y); {x is Trend, y is Residual}

   {MakeDataBox(residueBOX, residueVec); MakeDataBox(trendBOX, trendVec);}
    minResidual := y[0];
    maxResidual := y[0];
    trendMean := 0.0;
    residueMean := 0.0;
    for i := 0 to dataBOX.NOPUNCT - 1 do
      begin
        trendVector[i] := x[i];
        residueVec[i] := y[i];
        trendMean := trendMean + trendVector[i];
        residueMean := residueMean + residueVec[i];
        maxResidual := Rmax(maxResidual, y[i]);
        minResidual := Rmin(minResidual, y[i]);
      end;
    residueMean := residueMean / dataBOX.NOPUNCT;
    trendMean := trendMean / dataBOX.NOPUNCT;

    residueBOX := dataBOX;
    trendBOX := dataBOX;
    trendBOX.yMean := trendMean;
    residueBOX.yMean := residueMean;
    residueBOX.yMin := minResidual;
    residueBOX.yMax := maxResidual;

    weHaveATrendCycle := true;
    weHavePolyResiduals := false;
    weHaveTrendCycleResiduals := true;

  end; {MakeHPTrend}

{______________________________________________________________________________}

  procedure HPFilter (Tcap: integer;
                  lambda: real;
                  var x: longVector;
                  var y: longVector);

    var
      wVec, qqVec: vector;

  begin
    x := y;
    FormMvec(qqVec, 2); {Q'Q}
    FormQuinDiag(wVec, lambda);
    QprimeY(x, Tcap); { d = Q'y: we are loosing two dimesions in taking 2nd differences}
    Toeplitz(wVec, x, Tcap - 3, 2); {solve (I + lambda*Q'Q)z = d for z}
    FormQy(x, Tcap); {Form Qz}
    FormTrend(y, x, lambda, Tcap); {x = y - lambda *Qz}
  end; {HPFilter}

{______________________________________________________________________________}

  procedure Toeplitz (gamma: vector;
                  var y: longVector;
                  n, q: integer);

  {This procedure uses a Cholesky decomposition to find the solution of}
  {the equation Gx = y,  where  G is a symmetric Toeplitz matrix of order}
  {n with q supra-diagonal and q sub-diagonal bands . The coefficients of}
  {G are contained in gamma. The RHS vector y contains the elements of}
  {the solution vector x on completion.}

    var
      t, j, k: integer;
      mu: array[0..maxOrder, 0..maxArray] of real; {wideMatrix}

  begin {Toeplitz}

  {Factorise}
    for t := 0 to q - 1 do
      for j := t + 1 to q do
        mu[j, t] := 0.0;

    for t := 0 to n do
      begin {t}
        for k := Min(q, t) downto 0 do
          begin {k}
            mu[k, t] := gamma[k];
            for j := 1 to q - k do
              mu[k, t] := mu[k, t] - mu[j, t - k] * mu[j + k, t] * mu[0, t - k - j];
            if k > 0 then
              mu[k, t] := mu[k, t] / mu[0, t - k];
          end;{k}
      end; {t}

  {ForwardSolve;}
    for t := 0 to n do
      for j := 1 to Min(t, q) do
        y[t] := y[t] - mu[j, t] * y[t - j];

  {Divide by the diagonal}
    for t := 0 to n do
      y[t] := y[t] / mu[0, t];

  {BackSolve;}
    for t := n downto 0 do
      for j := 1 to Min(q, n - t) do
        y[t] := y[t] - mu[j, t + j] * y[t + j];

  end; {Toeplitz}

{______________________________________________________________________________}

  procedure SumDifference (var theta: vector;
                  n: integer;
                  sign: integer);

 {This procedure finds the coefficients of the nth power of}
 {the summation operator or of the difference operator.}

    var
      j, q: integer;

  begin
    theta[0] := 1.0;
    for q := 1 to n do
      begin {q}
        theta[q] := 0.0;
        for j := q downto 1 do
          theta[j] := theta[j] + sign * theta[j - 1];
      end; {q}

  end; {SumDifference}

{______________________________________________________________________________}

 procedure FormGamma (var gamma: vector;
                 mu: vector;
                 r: integer);

{This procedure forms the Autcovariances  }
{of an rth-order moving-average process.}

   var
     j, k: integer;

 begin
   for j := 0 to r do
     begin {j}
       gamma[j] := 0.0;
       for k := 0 to r - j do
         gamma[j] := gamma[j] + mu[k] * mu[k + j];
     end; {j}
 end; {FormGamma}

{______________________________________________________________________________}

  procedure FormMvec (var mVec: vector;
                  n: integer);

    var
      mu: vector;

  begin
    sumDifference(mu, n, 1);
    FormGamma(mVec, mu, n);
  end; {FormMvec}

{______________________________________________________________________________}

  procedure FormWvec (var wVec: vector;
                  n: integer;
                  lamOne, lamTwo: real);

    var
      i: integer;
      sVec, mVec: vector;

  begin
    FormSVec(sVec, n); {sVec = Q'SQ where Q is the 2nd operator}
    FormMVec(mVec, n);
    for i := 0 to n do
      wVec[i] := lamTwo * mVec[i] + lamOne * sVec[i];
  end; {FormwVec}

{______________________________________________________________________________}

  procedure FormQuinDiag (var wVec: vector;
                  lambda: real);

    var
      mu: vector;

  begin
    sumDifference(mu, 2, -1);
    FormGamma(wVec, mu, 2);
    wVec[0] := 1.0 + lambda * wVec[0];
    wVec[1] := lambda * wVec[1];
    wVec[2] := lambda * wVec[2];
  end; {FormQuinDiag}

{______________________________________________________________________________}

  procedure VariQinDiag (n: integer;
                  var d, s, q, y: longvector);
    var
      t: integer;

  begin

  {Factorisation}
    d[-2] := 0.0;
    d[-1] := 0.0;
    s[-1] := 0.0;
    q[-2] := 0.0;
    q[-1] := 0.0;
    for t := 0 to n do
      begin
        d[t] := d[t] - d[t - 1] * Sqr(s[t - 1]) - d[t - 2] * Sqr(q[t - 2]);
        s[t] := (s[t] - d[t - 1] * s[t - 1] * q[t - 1]) / d[t];
        q[t] := q[t] / d[t];
      end;

  {Forward Substitution}
    y[-2] := 0.0;
    y[-1] := 0.0;
    for t := 0 to n do
      y[t] := y[t] - s[t - 1] * y[t - 1] - q[t - 1] * y[t - 2];
    for t := 0 to n do
      y[t] := y[t] / d[t];

  {Back Substitution}
    y[n + 2] := 0.0;
    y[n + 1] := 0.0;
    for t := n downto 0 do
      y[t] := y[t] - s[t] * y[t + 1] - q[t] * y[t + 2];

  end;{VariQinDiag}

  {________________________________________________}

  procedure QprimeY (var y: longVector;
                  Tcap: integer);

 {This procedure finds the second differences of the elements of a vector.}
 {Notice that the first two elements of the vector are lost in the process.}
 {However, we index the leading element of the differenced vector by t = 0.}

    var
      t: integer;

  begin
    for t := 0 to Tcap - 3 do
      y[t] := y[t] - 2 * y[t + 1] + y[t + 2];
  end; {QprimeY}

{______________________________________________________________________________}

  procedure FormQy (var y: longVector;
                  Tcap: integer);

 {This procedure multiplies the vector y of Tcap -2 elements by matrix}
 {Q of order Tcap times Tcap -2, where Q' is the matrix which finds}
 {the second differences of a vector.}

    var
      t: integer;
      lag1, lag2, store: real;

  begin
    lag1 := 0.0;
    lag2 := 0.0;
    for t := 0 to Tcap - 3 do
      begin {t}
        store := y[t];
        y[t] := y[t] - 2 * lag1 + lag2;
        lag2 := lag1;
        lag1 := store
      end; {t}
    y[Tcap - 2] := -2 * lag1 + lag2;
    y[Tcap - 1] := lag1;

  end; {FormQy}

{______________________________________________________________________________}

  procedure FormSVec (var sVec: vector;
                  n: integer);

    var
      mu: vector;

  begin
    sumDifference(mu, n, -1);
    FormGamma(sVec, mu, n);
  end; {FormSvec}

{______________________________________________________________________________}

  procedure GammaY (gamma: vector;
                  var y: longVector;
                  Tcap, n: integer);

 {This procedure premultiplies a vector y by a symmetric banded Toeplitz matrix}
 {Gamma with n nonzero sub-diagonal bands and n nonzero supra-dagonal bands.}

    var
      i, t, j: integer;
      left, right: real;
      store: vector;

  begin
    for i := 0 to n do
      store[i] := 0.0;

    for t := 0 to Tcap - 1 do
      begin {t}
        for i := n downto 1 do
          store[i] := store[i - 1];
        store[0] := gamma[0] * y[t];
        for j := 1 to n do
          begin {j}
            if (t - j < 0) then
              left := 0
            else
              left := y[t - j];
            if (t + j > Tcap - 1) then
              right := 0
            else
              right := y[t + j];
            store[0] := store[0] + gamma[j] * (left + right);
          end; {j}
        if t >= n then
          y[t - n] := store[n];
      end; {t}
    for j := 0 to n - 1 do
      y[Tcap - 1 - j] := store[j];
  end; {GammaY}

{______________________________________________________________________________}

  procedure FormTrend (var y: longVector;
                  var x: longVector;
                  lamOne: real;
                  Tcap: integer);

    var
      t: integer;
      store: real;

  begin

    for t := 0 to Tcap - 1 do
      begin
        store := lamOne * x[t];
        x[t] := y[t] - store; {the trend vector}
        y[t] := store; {the residual vector}
      end;
  end; {FormTrend}

{______________________________________________________________________________}

end.  {WIENER: unit of SEASCAPE.PAS}

