unit MATHS; {Unit of SEASCAPE.PAS}

interface

  uses
    GLOBALS;

  function Even (q: integer): boolean;

  function Max (p, q: integer): integer;

  function Min (p, q: integer): integer;

  function Rmax (x, y: real): real;

  function Rmin (x, y: real): real;

  function Power (x: real;
                  n: integer): real;

  function RealPower (x: real;
                n: real): real;

  function Tan (theta: real): real;

  function CoTan (theta: real): real;

  function Sign (alpha: real): integer;

  function Arg (psi: complex): real;

  function TheArgument (thepoint: complex): real;

  function TheModulus (apoint: complex): real;

  procedure ComplexToPolar (complexPoint: complex;
                  var polarPoint: polar);

  procedure PolarToComplex (polarPoint: polar;
                  var complexPoint: complex);

  procedure QuadraticRoots (a, b, c: real;
                  var x1, x2: complex);

  procedure GaussianInversion (n, stop: integer;
                  var a: matrix);

  procedure LUsolve (start, n: integer;
                  var a: matrix;
                  var x, b: vector);

implementation

{_________________________________________________}

  function Even (q: integer): boolean;

  begin
    Even := false;
    if q mod 2 = 0 then
      Even := true;
  end; {Even}

{_________________________________________________}

  function Max (p, q: integer): integer;

  begin
    if p > q then
      Max := p
    else
      Max := q
  end; {Max}

{_________________________________________________}

  function Min (p, q: integer): integer;

  begin
    if p < q then
      Min := p
    else
      Min := q
  end; {Min}

{_________________________________________________}

  function Rmax (x, y: real): real;

  begin
    if x > y then
      Rmax := x
    else
      Rmax := y;
  end; {Rmax}

{_________________________________________________}

  function Rmin (x, y: real): real;

  begin
    if x > y then
      Rmin := y
    else
      Rmin := x;
  end; {Rmin}

{_________________________________________________}

  function Power (x: real;
                  n: integer): real;

    var
      logPower: real;

  begin
    x := Abs(x);
    if x < 1E-35 then
      Power := 0
    else
      begin {else}
        logPower := n * Ln(x);
        if (logPower < -80) then
          Power := 0
        else if (logPower > 80) then
          Power := Exp(80)
        else
          Power := Exp(logPower);
      end; {else}
  end; {Power}

{______________________________________________________________________________}

function RealPower (x: real;
                n: real): real;

  var
    logPower: real;

begin
  x := Abs(x);
  if x < 1E-35 then
    RealPower := 0
  else
    begin {else}
      logPower := n * Ln(x);
      if (logPower < -80) then
        RealPower := 0
      else if (logPower > 80) then
        RealPower := Exp(80)
      else
        RealPower := Exp(logPower);
    end; {else}
end; {RealPower}

{______________________________________________________________________________}

  function Tan (theta: real): real;

    var
      c, s: real;

  begin
    c := Cos(theta);
    s := Sin(theta);
    Tan := s / c;
  end; {Tan}

{_________________________________________________}

  function CoTan (theta: real): real;

    const
      almostZero = 1E-35;

    var
      c, s: real;

  begin
    c := Cos(theta);
    s := Sin(theta);
    if Abs(s) < almostZero then
      s := Sign(s) * almostZero;
    CoTan := c / s;
  end; {CoTan}

{_________________________________________________}

  function Sign (alpha: real): integer;

  begin
    Sign := 0;
    if alpha < 0 then
      Sign := -1
    else
      Sign := 1;
  end; {Sign}

{_________________________________________________}

  function Arg (psi: complex): real;

    var
      theta: real;

  begin
    if psi.re > 0 then
      theta := Arctan(psi.im / psi.re)
    else if (psi.re < 0) and (psi.im > 0) then
      theta := pi + Arctan(psi.im / psi.re)
    else if (psi.re < 0) and (psi.im < 0) then
      theta := -pi + Arctan(psi.im / psi.re);
    Arg := theta
  end; {Arg}

{_________________________________________________}

  function TheArgument (thepoint: complex): real;

    var
      theta: real;

  begin
    if thepoint.re = 0 then
      theta := Sign(thepoint.im) * pi / 2;
    if thepoint.re <> 0 then
      theta := Arctan(thepoint.im / thepoint.re);
    if thepoint.re < 0 then
      theta := theta + Sign(thepoint.im) * pi;
    TheArgument := theta;
  end; {TheArgument}

{_________________________________________________}

  function TheModulus (apoint: complex): real;
  begin
    TheModulus := Sqrt(apoint.re * apoint.re + apoint.im * apoint.im);
  end; {TheModulus}

  {_________________________________________________}

  procedure ComplexToPolar (complexPoint: complex;
                  var polarPoint: polar);
  begin
    polarPoint.modu := TheModulus(complexPoint);
    polarPoint.argu := TheArgument(complexPoint);
  end; {ComplexToPolar}

  {_________________________________________________}

  procedure PolarToComplex (polarPoint: polar;
                  var complexPoint: complex);
  begin
    complexPoint.re := polarPoint.modu * Cos(polarPoint.argu);
    complexPoint.im := polarPoint.modu * Sin(polarPoint.argu);
  end; {PolarToComplex}

{_________________________________________________}

  procedure ComplexSqrt (a: complex;
                  var result: complex);

    var
      b: complex;
      rho: real;

  begin
    rho := Sqrt(a.re * a.re + a.im * a.im);
    b.re := Sqrt((rho + a.re) / 2);
    b.im := Sign(a.im) * Sqrt((rho - a.re) / 2);
    result := b;
  end; {ComplexSqrt}

{_________________________________________________}

  procedure CpolyValue (a: complexvector;
                  p: integer;
                  z: complex;
                  var f: complex;
                  var b: complexvector);

    var
      i: integer;

  begin {CpolyValue}
    b[p - 1].re := a[p].re;
    b[p - 1].im := a[p].im;
    for i := 1 to p - 1 do
      begin
        b[p - i - 1].re := b[p - i].re * z.re - b[p - i].im * z.im + a[p - i].re;
        b[p - i - 1].im := b[p - i].im * z.re + b[p - i].re * z.im + a[p - i].im;
      end;
    f.re := b[0].re * z.re - b[0].im * z.im + a[0].re;
    f.im := b[0].im * z.re + b[0].re * z.im + a[0].im;
  end; {CpolyValue}

{_________________________________________________}

  procedure QuadraticRoots (a, b, c: real;
                  var x1, x2: complex);

    var
      delta: real;

  begin
    x1.im := 0;
    x2.im := 0;
    delta := Sqr(b) - 4 * a * c;
    if (delta > 0) and (a <> 0) then
      begin
        x1.re := (-b + Sqrt(delta)) / (2 * a);
        x2.re := (-b - Sqrt(delta)) / (2 * a);
      end;
    if (delta = 0) and (a <> 0) then
      begin
        x1.re := -b / (2 * a);
        x1.re := -b / (2 * a);
      end;
    if (delta < 0) and (a <> 0) then
      begin
        x1.re := -b / (2 * a);
        x1.im := Sqrt(-delta) / (2 * a);
        x2.re := -b / (2 * a);
        x2.im := -Sqrt(-delta) / (2 * a);
      end;
  end; {QuadraticRoots}

{_________________________________________________}

  procedure GaussianInversion (n, stop: integer;
                  var a: matrix);

 {This has been modified to ondex the lemsnts from  zero}
    var
      lambda, pivot: real;
      i, j, k: integer;

  begin {Gaussian Inversion}
    for i := 0 to stop do
      begin {i}
        pivot := a[i, i];
        for k := 0 to n do
          begin {k}
            if k <> i then
              begin
                lambda := a[k, i] / pivot;
                for j := 0 to n do
                  a[k, j] := a[k, j] - lambda * a[i, j];
                a[k, i] := -lambda
              end;
          end; {k}
        for j := 1 to n do
          a[i, j] := a[i, j] / pivot;
        a[i, i] := 1 / pivot;
      end; {i; reduction completed}

  end; {GaussianInversion}

{_________________________________________________}

  procedure LUsolve (start, n: integer;
                  var a: matrix;
                  var x, b: vector);

    type
      ivector = array[0..20] of integer;

    var
      v, w, pivot, lambda: real;
      i, j, k, pivotRow, finish: integer;
      p: ivector;
      d: vector;

  begin {LUsolve}

    finish := start + n - 1;
    for i := start to finish do
      begin {i; determine the scale factors}
        p[i] := i;
        d[i] := 0.0;
        for j := start to finish do
          if d[i] < Abs(a[i, j]) then
            d[i] := Abs(a[i, j]);
      end; {i}

    for i := start to finish - 1 do
      begin {i; begin the process of reduction}
        pivot := a[p[i], i];
        for k := i + 1 to finish do
          begin { k; search for a better pivot}
            v := Abs(pivot) / d[p[i]];
            w := Abs(a[p[k], i]) / d[p[k]];
            if v < w then
              begin  {interchange rows if a better pivot is found}
                pivot := a[p[k], i];
                pivotRow := p[k];
                p[k] := p[i];
                p[i] := pivotRow;
              end; {end interchange}
          end; {k; end the search for a pivot}

        for k := i + 1 to finish do
          begin {k; eliminate a[k, i]}
            lambda := a[p[k], i] / pivot;
            for j := i + 1 to finish do
              a[p[k], j] := a[p[k], j] - lambda * a[p[i], j];
            a[p[k], i] := lambda   {save the multiplier }
          end; {k}
      end; {i; reduction completed}

    for i := start to finish do
      begin {i; forward substitution}
        x[i] := b[p[i]];
        for j := i - 1 downto start do
          x[i] := x[i] - a[p[i], j] * x[j];
      end; {i; forward substitution}

    for i := finish downto start do
      begin {i; back substitution}
        for j := i + 1 to finish do
          x[i] := x[i] - a[p[i], j] * x[j];
        x[i] := x[i] / a[p[i], i];
      end; {i; back substitution}

  end; {LUsolve}

{__________________________________________}

  procedure XToeplitz (gamma: vector;    {REDUNDANT}
                  var y: longVector;
                  n, q: integer);

{This procedure uses a Cholesky decomposition to find the solution of}
{the equation Gx = y,  where  G is a symmetric Toeplitz matrix of order}
{n with q supra-diagonal and q sub-diagonal bands . The coefficients of}
{G are contained in gamma. The RHS vector y contains the elements of}
{the solution vector x on completion.}

    var
      t, j, k: integer;
      mu: array[0..maxOrder, 0..maxArray] of real; {wideMatrix}

  begin {Toeplitz}

{Factorise}
    for t := 0 to q - 1 do
      for j := t + 1 to q do
        mu[j, t] := 0.0;

    for t := 0 to n do
      begin {t}
        for k := Min(q, t) downto 0 do
          begin {k}
            mu[k, t] := gamma[k];
            for j := 1 to q - k do
              mu[k, t] := mu[k, t] - mu[j, t - k] * mu[j + k, t] * mu[0, t - k - j];
            if k > 0 then
              mu[k, t] := mu[k, t] / mu[0, t - k];
          end;{k}
      end; {t}

{ForwardSolve;}
    for t := 0 to n do
      for j := 1 to Min(t, q) do
        y[t] := y[t] - mu[j, t] * y[t - j];

{Divide by the diagonal}
    for t := 0 to n do
      y[t] := y[t] / mu[0, t];

{BackSolve;}
    for t := n downto 0 do
      for j := 1 to Min(q, n - t) do
        y[t] := y[t] - mu[j, t + j] * y[t + j];

  end; {Toeplitz}

{_________________________________________________}

end. {MATHS: Unit of SEASCAPE.PAS}

