unit DATASEG; {Unit of SEASCAPE.PAS}

interface

  uses
    GLOBALS, UTILS, MATHS, Dialogs;


  procedure LogOfVector (var dataBOX: rBOX;
                  var dataVector: longVector);

  procedure DiffOfVector (var dataBOX: rBOX;
                    var dataVector: longVector);

  procedure AntiLogOfVector (var dataBOX: rBOX;
                  var dataVector: longVector);

  procedure MeanAdjustment (var dataBOX: rBOX;
                    var dataVector: longVector);

  procedure PolyRegress (var x, y: longVector;        {TO REGRESS}
                  var alpha, gamma, delta: vector;
                  var poly, wgt: longVector;
                  q, n: integer);

  procedure ExePolyRegress (var x, y: longVector;        {TO REGRESS}
                  var alpha, gamma, delta: vector;
                  var poly, wgt: longVector;
                  q, n: integer);

  procedure OrthoToPower (alpha, gamma, delta: vector;      {TO REGRESS}
                  var beta: vector;
                  q: integer);

  function PolyOrdinate (z: real;
                  var alpha, gamma, delta: vector;      {TO REGRESS}
                  q: integer): real;

  procedure MakePolyTrend (var polyParas: polyParameters;      {TO REGRESS}
                  var dataBOX, trendBox, residueBOX: rBOX;
                  var dataVector, trendVector, residueVec, PolyWgts: longVector);

  procedure QuadraticWgts (var PolyWgts: longVector;       {TO REGRESS}
                  var minWgt, maxWgt,midRange: real;
                  var dataBOX: rBOX);

  procedure CrenellatedWgts (var PolyWgts: longVector;    {TO REGRESS}
                  var minWgt, maxWgt,midRange: real;
                  var dataBOX: rBOX);

  procedure CosBellWgts (var PolyWgts: longVector;     {TO REGRESS}
                  var minWgt, maxWgt, midRange: real;
                  var dataBOX: rBOX);

  procedure UniformWgts (var PolyWgts: longVector;      {TO REGRESS}
                  var dataBOX: rBOX);

  procedure SlowFourier (var dataBOX: rBOX;
                  var dataVector: longVector;
                  var alpha: fourierVec;
                  var beta: fourierVec);

  procedure FourierSynthesis (var dataBOX: rBOX;
                  var dataVector: longVector;
                  var alpha: fourierVec;
                  var beta: fourierVec);

  procedure BandPassCoefficients (var dataBOX: rBOX;
                  var dataVector: longVector;
                  FourierInc: real;
                  lowerCutInt, upperCutInt: integer;
                  var alpha: fourierVec;
                  var beta: fourierVec);

  procedure BandPassSynthesis (var dataBOX: rBOX;
                    var dataVector: longVector;
                    fourierInc: real;
                    lowerCutInt, upperCutInt: integer;
                    var alpha: fourierVec;
                    var beta: fourierVec);

  procedure MakeThePeriodogram (var dataBOX: rBOX;
                  var spectraBOX: rBOX;
                  var spectraVec: longVector;
                  var alpha, beta: fourierVec);

  procedure TrendCycleComponent (var polyParas: polyParameters;
                  var trendBOX, cycleBOX, residueBOX: rBOX;
                  var trendVector, cycleVec, residueVec: longVector);

  procedure SubtractComponent (var dataBOX, adjustedBOX: rBOX;
                      var dataVector, componentVec, adjustedVec: longVector);

implementation

{________________________________________________}

  procedure LogOfVector (var dataBOX: rBOX;
                  var dataVector: longVector);

    var
      i: integer;
      logDataVec: longVector;
      badElements: boolean;

  begin
    i := 0;
    badElements := false;
    while (i < dataBOX.NOPUNCT) and (badElements = false) do
      begin {while}
        if dataVector[i] <= 0 then
          begin
            badElements := true;
            logsTaken := false;
          end
        else
          begin
            logDataVec[i] := ln(dataVector[i]);
            i := i + 1;
          end;
      end; {while}

    ResetGlobalFlags;

    if badElements = false then
      begin
        dataVector := logDataVec;
        MakeDataBox(dataBOX, dataVector);
        logsTaken := true;
      end;

  end; {LogOfVector}

{______________________________________________________________________________}

    procedure DiffOfVector (var dataBOX: rBOX;
                    var dataVector: longVector);

     {dataBOX.NOPUNCT is reduced by one}

      var
        t: integer;

    begin
     for t := 0 to dataBOX.NOPUNCT - 2 do
          dataVector[t] := dataVector[t + 1] - dataVector[t];

      dataBOX.NOPUNCT := dataBOX.NOPUNCT - 1;
      dataBOX.diffOrder := dataBOX.diffOrder + 1;
      MakeDataBox (dataBOX,dataVector);

      ResetGlobalFlags;

    end; {DiffOfVector}

{______________________________________________________________________________}

  procedure AntiLogOfVector (var dataBOX: rBOX;
                  var dataVector: longVector);

    var
      i: integer;

  begin

    with dataBOX do
      begin
        yMax := Exp(dataVector[0]);
        yMin := yMax;
        yMean := 0.0;
        for i := 0 to NOPUNCT - 1 do
          begin {i}
            dataVector[i] := Exp(dataVector[i]);
            if dataVector[i] > yMax then
              yMax := dataVector[i];
            if dataVector[i] < yMin then
              yMin := dataVector[i];
            yMean := yMean + dataVector[i];
          end; {i}
        yMean := yMean / dataBOX.NOPUNCT;
      end; {with DataBOX}

    logsTaken := false;

  end; {AntiLogOfVector}

{______________________________________________________________________________}

  procedure MeanAdjustment (var dataBOX: rBOX;
                    var dataVector: longVector);
    var
      t: integer;

    begin
      for t := 0 to dataBOX.NOPUNCT - 1 do
       dataVector[t] := dataVector[t]- dataBOX.yMean;

      MakeDataBox (dataBOX, dataVector);
      meanAdjustedData := true;
      ResetGlobalFlags;

    end;   {MeanAdjustment}

{______________________________________________________________________________}

  procedure PolyRegress (var x, y: longVector;
                  var alpha, gamma, delta: vector;
                  var poly, wgt: longVector;
                  q, n: integer);

    var
      phi, philag: longVector;
      denom, lagdenom, temp: real;
      k, t: integer;

  begin {PolyRegress}

    alpha[0] := 0.0;
    gamma[1] := 0.0;
    denom := 0.0;

    for t := 0 to n do
      begin
        alpha[0] := alpha[0] + y[t] * wgt[t];
        gamma[1] := gamma[1] + x[t] * wgt[t];
        denom := denom + wgt[t];
      end;

    alpha[0] := alpha[0] / denom;
    gamma[1] := gamma[1] / denom;
    lagdenom := denom;

    for t := 0 to n do
      begin
        philag[t] := 1;
        phi[t] := x[t] - gamma[1];
        poly[t] := alpha[0];
      end;

    for k := 1 to q do
      begin {k}

        alpha[k] := 0.0;
        gamma[k + 1] := 0;
        denom := 0.0;

        for t := 0 to n do
          begin {t}
            alpha[k] := alpha[k] + y[t] * phi[t] * wgt[t];
            denom := denom + Sqr(phi[t]) * wgt[t];
            gamma[k + 1] := gamma[k + 1] + Sqr(phi[t]) * x[t] * wgt[t];
          end; {t}

        alpha[k] := alpha[k] / denom;
        gamma[k + 1] := gamma[k + 1] / denom;
        delta[k + 1] := denom / lagdenom;
        lagdenom := denom;

        for t := 0 to n do
          begin {t}
            poly[t] := poly[t] + alpha[k] * phi[t];
            temp := phi[t];
            phi[t] := (x[t] - gamma[k + 1]) * phi[t] - delta[k + 1] * philag[t];
            philag[t] := temp;
          end; {t}

      end; {k}

  end; {PolyRegress}

  {________________________________________________}

  procedure ExePolyRegress (var x, y: longVector;
                  var alpha, gamma, delta: vector;
                  var poly, wgt: longVector;
                  q, n: integer);

  {It looks  as if this is now redundant}

    var
      phi, philag: longVector;
      denom, lagdenom, temp: real;
      k, t: integer;

  begin {PolyRegress}

    alpha[0] := 0.0;
    gamma[1] := 0.0;
    denom := 0.0;

    for t := 0 to n do
      begin
        alpha[0] := alpha[0] + y[t] * wgt[t];
        gamma[1] := gamma[1] + x[t] * wgt[t];
        denom := denom + wgt[t];
      end;

    alpha[0] := alpha[0] / denom;
    gamma[1] := gamma[1] / denom;
    lagdenom := denom;

    for t := 0 to n do
      begin
        philag[t] := 1;
        phi[t] := x[t] - gamma[1];
        poly[t] := alpha[0];
      end;

    for k := 1 to q do
      begin {k}

        alpha[k] := 0.0;
        gamma[k + 1] := 0;
        denom := 0.0;

        for t := 0 to n do
          begin {t}
            alpha[k] := alpha[k] + y[t] * phi[t] * wgt[t];
            denom := denom + Sqr(phi[t]) * wgt[t];
            gamma[k + 1] := gamma[k + 1] + Sqr(phi[t]) * x[t] * wgt[t];
          end; {t}

        alpha[k] := alpha[k] / denom;
        gamma[k + 1] := gamma[k + 1] / denom;
        delta[k + 1] := denom / lagdenom;
        lagdenom := denom;

        for t := 0 to n do
          begin {t}
            poly[t] := poly[t] + alpha[k] * phi[t];
            temp := phi[t];
            phi[t] := (x[t] - gamma[k + 1]) * phi[t] - delta[k + 1] * philag[t];
            philag[t] := temp;
          end; {t}

      end; {k}

  end; {ExePolyRegress}

  {________________________________________________}

  procedure OrthoToPower (alpha, gamma, delta: vector;
                  var beta: vector;
                  q: integer);

    var
      phiplus: real;
      phi, philag: vector;
      i, k: integer;

  begin
    phi[-1] := 0;
    phi[0] := -gamma[1];
    phi[1] := 1; {redundant}
    philag[0] := 1;
    philag[1] := 0;
    beta[0] := alpha[0];

  {find the power-form coefficients}
    for k := 1 to q do
      begin {k}
        beta[k] := 0.0;
        for i := k downto 0 do
          begin
            beta[i] := beta[i] + alpha[k] * phi[i];
            phiplus := phi[i - 1] - gamma[k + 1] * phi[i] - delta[k + 1] * philag[i];
            philag[i] := phi[i];
            phi[i] := phiplus;
          end;
        phi[k + 1] := 1;
        philag[K + 1] := 0;
      end;{k}

  end; {OrthoToPower}

  {________________________________________________}

  function PolyOrdinate (z: real;
                  var alpha, gamma, delta: vector;
                  q: integer): real;

    {alpha, gamma, delta and q--> polyDegree should be subsumbed under polyParas}
{z is the absicissa, alpha[j] are parameters  of the power form of the polynomial}
    var
      d: vector;
      j: integer;

  begin
    d[q] := alpha[q];
    d[q - 1] := alpha[q - 1] + d[q] * (z - gamma[q]);
    for j := q - 2 downto 0 do
      d[j] := alpha[j] + d[j + 1] * (z - gamma[j + 1]) - d[j + 2] * delta[j + 2];
    PolyOrdinate := d[0];

  end; {PolyOrdinate}

  {________________________________________________}

  procedure MakePolyTrend (var polyParas: polyParameters;
                  var dataBOX, trendBox, residueBOX: rBOX;
                  var dataVector, trendVector, residueVec, PolyWgts: longVector);

 {Called by FitPolyTrend in DATASEG: alpha, gamma, delta and polyOrder--> polyDegree}
 {are subsumbed under polyParas}

    var
      x: longVector;
      i: integer;

  begin
    with PolyParas do
      begin {polyParas}
        tStart := 0;
        residueBOX := dataBOX;
        trendBOX := dataBOX;

        for i := 0 to dataBOX.NOPUNCT - 1 do
          x[i] := i;

        PolyRegress(x, dataVector, alpha, gamma, delta, trendVector, PolyWgts, polyDegree, dataBOX.NOPUNCT - 1);

        if polyDegree < 10 then
          OrthoToPower(alpha, gamma, delta, beta, polyDegree);

        for i := 0 to dataBOX.NOPUNCT - 1 do
          residueVec[i] := dataVector[i] - trendVector[i];

      end; {polyParas}

    MakeDataBox(trendBOX, trendVector);
    MakeDataBox(residueBOX, residueVec);
    {trendEstimated := true;}   {NOT IN USE}
    weHavePolyResiduals := true;
    weHaveATrendCycle := false;


  end; {MakePolyTrend}

  {_______________________________________________}

  procedure QuadraticWgts (var PolyWgts: longVector;
                  var minWgt, maxWgt, midRange: real;
                  var dataBOX: rBOX);
    var
      f0, f1, z1, z2, a, b, c: real;
      det, quadratic: real;
      t, Tcap, cutOff: integer;

  begin

     Tcap := dataBOX.NOPUNCT;
     if (midRange) > 0.0 then
      begin
        midRange := midRange / 100.0;
        cutOff := Round(Tcap * (1.0 - MidRange) / 2.0);
        z1 := cutOff;
        z2 := 2 * cutOff;
      end
    else
      begin
        cutoff := Tcap div 2;
        z1 := (Tcap - 1) / 2.0;
        z2 := (Tcap - 1);
      end;

  {Find the quadratic parameters}

    f0 := maxWgt;
    f1 := minWgt;
    c := f0;
    det := z1 * (z1 * z2 - z2 * z2);
    det := 1.0 / det;
    c := f0;
    a := z2 * (f1 - f0) * det;
    b := -z2 * a;

    for t := 0 to Tcap div 2 do
      begin
        if (t <= cutoff) then
          begin
            quadratic := a * t + b;
            quadratic := quadratic * t + c;
          end
        else
          quadratic := minWgt;
        PolyWgts[t] := quadratic;
        PolyWgts[Tcap - 1 - t] := quadratic;
      end; {t}

    dataBOX.ymax := maxWgt;
    dataBOX.ymin := minWgt;


  end; {QuadraticWgts}

  {_______________________________________________}

  procedure CrenellatedWgts (var PolyWgts: longVector;
                  var minWgt, maxWgt, midRange: real;
                  var dataBOX: rBOX);
    var
      crenWeight: real;
      t, Tcap, cutOff: integer;

  begin

    Tcap := dataBOX.NOPUNCT;

    if (Midrange) > 0.0 then
      begin
        midRange := Midrange / 100.0;
        cutOff := Round(Tcap * (1.0 - MidRange) / 2.0);
      end
    else
      cutoff := Tcap div 4;

    for t := 0 to Tcap div 2 do
      begin
        if (t <= cutoff) then
          begin
            crenWeight := MaxWgt;
          end
        else
          crenWeight := MinWgt;
        PolyWgts[t] := crenWeight;
        PolyWgts[Tcap - 1 - t] := crenWeight;
      end; {t}

    dataBOX.ymax := maxWgt;
    dataBOX.ymin := minWgt;

  end; {CrenellatedWgts}

  {_______________________________________________}

  procedure CosBellWgts (var PolyWgts: longVector;
                  var minWgt, maxWgt, midRange: real;
                  var dataBOX: rBOX);
    var
       pi, omega, gamma, cosWeight: real;
      t, Tcap, cutOff: integer;

  begin

    pi := 4.0 * ArcTan(1.0);
    Tcap := dataBOX.NOPUNCT;
    if (Midrange) > 0.0 then
      begin
        midRange := Midrange / 100.0;
        cutOff := Round(Tcap * (1.0 - MidRange) / 2.0);
      end
    else
      cutoff := Tcap div 2;

    for t := 0 to Tcap div 2 do
      begin
        if (t <= cutoff) then
          begin
            omega := (t * pi) / cutOff;
            gamma := (maxWgt - minWgt) / 2.0;
            cosWeight := minWgt + gamma * (1 + Cos(omega));
          end
        else
          cosWeight := minWgt;
        PolyWgts[t] := cosWeight;
        PolyWgts[Tcap - 1 - t] := cosWeight;
      end; {t}

    dataBOX.ymax := maxWgt;
    dataBOX.ymin := minWgt;

  end; {CosBellWgts}

 {_______________________________________________}

  procedure UniformWgts (var PolyWgts: longVector;
                  var dataBOX: rBOX);

    var
      t, Tcap: integer;

  begin
    Tcap := dataBOX.NOPUNCT;
    for t := 0 to Tcap - 1 do
      PolyWgts[t] := 1.0;

  end; {UniformWgts}

{________________________________________________}

  procedure SlowFourier (var dataBOX: rBOX;
                  var dataVector: longVector;
                  var alpha: fourierVec;
                  var beta: fourierVec);
    var
      sign, j, n, t, Tcap: integer;
      pi, FourierInc, omegaj, omegajt: real;

  begin

    Tcap := dataBOX.NOPUNCT;
    n := Tcap div 2;
    pi := 4.0 * Arctan(1.0);

   if Tcap = 0
   then Tcap := PUNCTSTORE;

    FourierInc := 2.0 * pi / Tcap;

    alpha[0] := 0.0;
    beta[0] := 0.0;
    alpha[n] := 0.0;
    beta[n] := 0.0; {in case Tcap is even}

    for t := 0 to Tcap - 1 do
      begin
        alpha[0] := alpha[0] + dataVector[t];
      end;
    alpha[0] := alpha[0] / Tcap;

    if Even(Tcap) then
      begin
        sign := 1;
        for t := 0 to Tcap - 1 do
          begin
            alpha[n] := alpha[n] + sign * dataVector[t];
            sign := -sign;
          end;
        alpha[n] := alpha[n] / Tcap;
        n := n - 1;
      end;

    for j := 1 to n do
      begin {j}
        omegaj := FourierInc * j;
        alpha[j] := 0.0;
        beta[j] := 0.0;
        for t := 0 to Tcap - 1 do
          begin
            omegajt := omegaj * t;
            alpha[j] := alpha[j] + Cos(omegajt) * dataVector[t];
            beta[j] := beta[j] + Sin(omegajt) * dataVector[t];
          end;
        alpha[j] := (alpha[j] * 2.0) / Tcap;
        beta[j] := (beta[j] * 2.0) / Tcap;
      end; {j}

  end; {SlowFourier}

{________________________________________________}

  procedure FourierSynthesis (var dataBOX: rBOX;
                  var dataVector: longVector;
                  var alpha: fourierVec;
                  var beta: fourierVec);

    var
      t, j, Tcap, n: integer;
      pi, fourierInc, omegat, omegajt, yMax, yMin: real;

  begin

    Tcap := dataBOX.NOPUNCT;
    n := Tcap div 2;

    pi := 4.0 * Arctan(1.0);
    fourierInc := 2 * pi / Tcap;

    yMax := 0.0;
    yMin := 0.0;

    for t := 0 to Tcap - 1 do
      begin {t}
        dataVector[t] := alpha[0];
        omegat := fourierInc * t;
        for j := 1 to n do
          begin {j}
            omegajt := omegat * j;
            dataVector[t] := dataVector[t] + alpha[j] * Cos(omegajt);
            dataVector[t] := dataVector[t] + beta[j] * Sin(omegajt);
          end; {j}
      end;{t}


    {dataBOX.yMax := yMax;
    dataBOX.yMin := yMin;
    dataBOX.xMax := dataBOX.NOPUNCT - 1;  {the maximum value of x in the dataVector}
    dataBOX.xMin := 0.0;   {the minimum value of x in the dataVector}
    dataBOX.xLB := dataBOX.xMin;   {xLB <= xMin: a left (lower) bound for x}
    dataBOX.xRB := dataBOX.xMax;    {xRB >= xMax: a right (upper) bound for x}}


  MakeDataBox(dataBOX, dataVector);

  end; {FourierSynthesis}

{______________________________________________________________________________}

  procedure BandPassCoefficients (var dataBOX: rBOX;
                  var dataVector: longVector;
                  FourierInc: real;
                  lowerCutInt, upperCutInt: integer;
                  var alpha: fourierVec;
                  var beta: fourierVec);

    var
      t, j: integer;
      omegaj, omegajt: real;

  begin

    for j := 0 to upperCutInt do
      begin
        alpha[j] := 0.0;
        beta[j] := 0.0;
      end; {This is unnecssary}

    alpha[0] := 0.0;
    if lowerCutInt = 0 then
      begin
        for t := 0 to dataBOX.NOPUNCT - 1 do
          alpha[0] := alpha[0] + dataVector[t];
        alpha[0] := alpha[0] / dataBOX.NOPUNCT;
      end;

    for j := lowerCutInt to upperCutInt do
      begin {j}
        omegaj := FourierInc * j;
        alpha[j] := 0.0;
        beta[j] := 0.0;
        for t := 0 to dataBOX.NOPUNCT - 1 do
          begin {t}
            omegajt := omegaj * t;
            alpha[j] := alpha[j] + Cos(omegajt) * dataVector[t];
            beta[j] := beta[j] + Sin(omegajt) * dataVector[t];
          end; {t}
        alpha[j] := alpha[j] * 2 / dataBOX.NOPUNCT;
        beta[j] := beta[j] * 2 / dataBOX.NOPUNCT;
      end; {j}
    if (2 * upperCutInt) = dataBOX.NOPUNCT then
      begin
        j := upperCutInt;
        alpha[j] := alpha[j] / 2;
        beta[j] := 0.0;
      end;

  end; {BandPassCoefficients}

{_____________________________________________________________________________}

    procedure BandPassSynthesis (var dataBOX: rBOX;
                    var dataVector: longVector;
                    fourierInc: real;
                    lowerCutInt, upperCutInt: integer;
                    var alpha: fourierVec;
                    var beta: fourierVec);

      var
        t, j: integer;
        omegat, omegajt, yMax, yMin: real;

    begin
      yMax := 0.0;
      yMin := 0.0;

      for t := 0 to dataBOX.NOPUNCT - 1 do
        begin {t}
          omegat := fourierInc * t;
          dataVector[t] := 0.0;
          for j := lowerCutInt to upperCutInt do
            begin {j}
              omegajt := omegat * j;
              dataVector[t] := dataVector[t] + alpha[j] * Cos(omegajt);
              dataVector[t] := dataVector[t] + beta[j] * Sin(omegajt);
            end; {j}
          yMax := Rmax(ymax, dataVector[t]);
          yMin := Rmin(ymin, dataVector[t]);
        end;{t}

      dataBOX.yMax := yMax;
      dataBOX.yMin := yMin;
      dataBOX.xMax := dataBOX.NOPUNCT - 1;  {the maximum value of x in the dataVector}
      dataBOX.xMin := 0.0;   {the minimum value of x in the dataVector}
      dataBOX.xLB := dataBOX.xMin;   {xLB <= xMin: a left (lower) bound for x}
      dataBOX.xRB := dataBOX.xMax;    {xRB >= xMax: a right (upper) bound for x}

    end; {BandPassSynthesis}

{_____________________________________________________________________________}

  procedure MakeThePeriodogram (var dataBOX: rBOX;
                  var spectraBOX: rBOX;
                  var spectraVec: longVector;
                  var alpha, beta: fourierVec);

    var
      j, Tcap, Ncap: integer;

  begin

    Tcap := dataBOX.NOPUNCT;
    Ncap := Tcap div 2;
    spectraBOX.NOPUNCT := Ncap + 1;
    spectraBOX.yMax := 0.0;
    spectraBOX.yMin := 0.0;

    spectraVec[0] := 0.0; {Mean adjustment}
    for j := 1 to Ncap do
      begin
        spectraVec[j] := alpha[j] * alpha[j] + beta[j] * beta[j];
        spectraVec[j] := (spectraVec[j] * Tcap) / 2;
        spectraBOX.yMax := RMax(spectraBOX.yMax, spectraVec[j]);
      end;


  {SHOWMESSAGE( ' spectraBOX.yMax = '+ TalyToString(spectraBOX.yMax) + ' spectraBOX.yMin = '+ TalyToString(spectraBOX.yMin));}

  {observe that beta[0] = 0 and that, if T is even. then beta[Ncap] := 0}
  {These values have been determined within <SlowFourier>}

  end; {MakeThePeriodogram }

{______________________________________________________________________________}


  procedure TrendCycleComponent (var polyParas: polyParameters;
                  var trendBOX, cycleBOX, residueBOX: rBOX;
                  var trendVector, cycleVec, residueVec: longVector);

    var
      z, polyt: real;
      t: integer;

    {trendVec contains the trendcycles, cycleVec contains the data}

  begin

        for t := 0 to cycleBOX.NOPUNCT - 1 do
          begin {t}
            z := t;
            polyt := PolyOrdinate(z, polyParas.alpha, polyParas.gamma, polyParas.delta, polyParas.polyDegree);
            trendVector[t] := cycleVec[t] + polyt;
            residueVec[t] := dataVector[t] - trendVector[t];
          end; {t}

        MakeDataBox(trendBOX, trendVector);
        MakeDataBox(residueBOX, residueVec);
        weHaveATrendCycle := true;
        weHavePolyResiduals := false;
        weHaveTrendCycleResiduals:= true;

  end; {TrendCycleComponent}

{______________________________________________________________________________}

  procedure AddThePolynomial (var polyParas: polyParameters;
                    var trendBOX, residueBOX: rBOX;
                    var trendVector, residueVec: longVector);

  var
    z, polyt: real;
    t: integer;

  begin

    for t := 0 to residueBOX.NOPUNCT - 1 do
      begin {t}
        z := t;
        polyt := PolyOrdinate(z, polyParas.alpha, polyParas.gamma, polyParas.delta, polyParas.polyDegree);
        trendVector[t] := cycleVec[t] + polyt;
      end; {t}

      MakeDataBox(trendBOX, trendVector);

    end; {AddThePolynomial}

{______________________________________________________________________________}

  procedure SubtractComponent (var dataBOX, adjustedBOX: rBOX;
                     var dataVector, componentVec, adjustedVec: longVector);

  var
    t: integer;

  begin

    for t := 0 to dataBOX.NOPUNCT - 1 do
      adjustedVec[t] := dataVector[t]- componentVec[t];

    adjustedBOX := dataBOX;
    MakeDataBox(adjustedBOX, adjustedVec);

  end; {SubtractSeasons}

{______________________________________________________________________________}


end. {DATASEG: Unit of SEASCAPE.PAS}


